<?php
/**
 * Admin Posts Management Routes
 */

function createPost() {
    $user = JWT::verify();
    
    $input = json_decode(file_get_contents('php://input'), true);
    
    $required = ['title', 'content'];
    foreach ($required as $field) {
        if (empty($input[$field])) {
            http_response_code(400);
            echo json_encode([
                'success' => false,
                'message' => "Field '$field' is required"
            ]);
            return;
        }
    }
    
    $db = Database::getInstance()->getConnection();
    
    // Generate slug
    $slug = generateSlug($input['title'], $db);
    
    // Extract excerpt if not provided
    $excerpt = $input['excerpt'] ?? substr(strip_tags($input['content']), 0, 160);
    
    $query = "INSERT INTO posts (
                title, slug, content, excerpt, featured_image,
                category_id, author_id, status,
                meta_title, meta_description, meta_keywords,
                published_at
              ) VALUES (
                :title, :slug, :content, :excerpt, :featured_image,
                :category_id, :author_id, :status,
                :meta_title, :meta_description, :meta_keywords,
                :published_at
              )";
    
    $publishedAt = ($input['status'] ?? 'draft') === 'published' 
        ? date('Y-m-d H:i:s') 
        : null;
    
    $stmt = $db->prepare($query);
    $stmt->execute([
        ':title' => $input['title'],
        ':slug' => $slug,
        ':content' => $input['content'],
        ':excerpt' => $excerpt,
        ':featured_image' => $input['featured_image'] ?? null,
        ':category_id' => $input['category_id'] ?? null,
        ':author_id' => $user['user_id'],
        ':status' => $input['status'] ?? 'draft',
        ':meta_title' => $input['meta_title'] ?? $input['title'],
        ':meta_description' => $input['meta_description'] ?? $excerpt,
        ':meta_keywords' => $input['meta_keywords'] ?? null,
        ':published_at' => $publishedAt
    ]);
    
    $postId = $db->lastInsertId();
    
    // Handle tags
    if (!empty($input['tags'])) {
        saveTags($postId, $input['tags'], $db);
    }
    
    echo json_encode([
        'success' => true,
        'data' => [
            'id' => $postId,
            'slug' => $slug
        ],
        'message' => 'Post created successfully'
    ]);
}

function updatePost($id) {
    $user = JWT::verify();
    
    $input = json_decode(file_get_contents('php://input'), true);
    $db = Database::getInstance()->getConnection();
    
    // Check if post exists
    $checkQuery = "SELECT id FROM posts WHERE id = :id";
    $checkStmt = $db->prepare($checkQuery);
    $checkStmt->execute([':id' => $id]);
    
    if (!$checkStmt->fetch()) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Post not found'
        ]);
        return;
    }
    
    // Build update query dynamically
    $updates = [];
    $params = [':id' => $id];
    
    $allowedFields = [
        'title', 'content', 'excerpt', 'featured_image',
        'category_id', 'status', 'meta_title', 'meta_description', 'meta_keywords'
    ];
    
    foreach ($allowedFields as $field) {
        if (isset($input[$field])) {
            $updates[] = "$field = :$field";
            $params[":$field"] = $input[$field];
        }
    }
    
    // Update slug if title changed
    if (isset($input['title'])) {
        $slug = generateSlug($input['title'], $db, $id);
        $updates[] = "slug = :slug";
        $params[':slug'] = $slug;
    }
    
    // Update published_at if status changed to published
    if (isset($input['status']) && $input['status'] === 'published') {
        $updates[] = "published_at = COALESCE(published_at, NOW())";
    }
    
    if (empty($updates)) {
        http_response_code(400);
        echo json_encode([
            'success' => false,
            'message' => 'No valid fields to update'
        ]);
        return;
    }
    
    $query = "UPDATE posts SET " . implode(', ', $updates) . " WHERE id = :id";
    $stmt = $db->prepare($query);
    $stmt->execute($params);
    
    // Handle tags
    if (isset($input['tags'])) {
        saveTags($id, $input['tags'], $db);
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Post updated successfully'
    ]);
}

function deletePost($id) {
    $user = JWT::verify();
    
    $db = Database::getInstance()->getConnection();
    
    $query = "DELETE FROM posts WHERE id = :id";
    $stmt = $db->prepare($query);
    $stmt->execute([':id' => $id]);
    
    if ($stmt->rowCount() === 0) {
        http_response_code(404);
        echo json_encode([
            'success' => false,
            'message' => 'Post not found'
        ]);
        return;
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Post deleted successfully'
    ]);
}

// Helper functions
function generateSlug($title, $db, $excludeId = null) {
    $slug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $title), '-'));
    $originalSlug = $slug;
    $counter = 1;
    
    while (true) {
        $query = "SELECT id FROM posts WHERE slug = :slug";
        if ($excludeId) {
            $query .= " AND id != :exclude_id";
        }
        $stmt = $db->prepare($query);
        $params = [':slug' => $slug];
        if ($excludeId) {
            $params[':exclude_id'] = $excludeId;
        }
        $stmt->execute($params);
        
        if (!$stmt->fetch()) {
            break;
        }
        
        $slug = $originalSlug . '-' . $counter++;
    }
    
    return $slug;
}

function saveTags($postId, $tags, $db) {
    // Delete existing tags
    $deleteQuery = "DELETE FROM post_tags WHERE post_id = :post_id";
    $deleteStmt = $db->prepare($deleteQuery);
    $deleteStmt->execute([':post_id' => $postId]);
    
    foreach ($tags as $tagName) {
        $tagSlug = strtolower(trim(preg_replace('/[^A-Za-z0-9-]+/', '-', $tagName), '-'));
        
        // Insert or get tag
        $tagQuery = "INSERT INTO tags (name, slug) VALUES (:name, :slug) 
                     ON DUPLICATE KEY UPDATE id=LAST_INSERT_ID(id)";
        $tagStmt = $db->prepare($tagQuery);
        $tagStmt->execute([':name' => $tagName, ':slug' => $tagSlug]);
        $tagId = $db->lastInsertId();
        
        // Link tag to post
        $linkQuery = "INSERT INTO post_tags (post_id, tag_id) VALUES (:post_id, :tag_id)";
        $linkStmt = $db->prepare($linkQuery);
        $linkStmt->execute([':post_id' => $postId, ':tag_id' => $tagId]);
    }
}